# Keelance Design System

> Notion-inspired, app-like UI with compact styling. Use this guide to restyle pages consistently.

---

## 1. Theme Setup

Every page must have:

```html
<html lang="en" data-theme="<?= h($currentTheme) ?>">
```

Theme is passed via `$currentTheme` variable (usually from `$_GET['theme'] ?? 'dark'`).

---

## 2. Color Palette

### Dark Mode (Default)

| Role | Value |
|------|-------|
| **Page Background** | `#0f0f0f` |
| **Navigation Background** | `#191919` |
| **Page Header Background** | `#0f0f0f` |
| **Card/Surface Background** | `rgba(255,255,255,0.03)` |
| **Card Hover** | `rgba(255,255,255,0.05)` |
| **Border/Divider** | `rgba(255,255,255,0.04)` to `rgba(255,255,255,0.08)` |
| **Text Primary** | `rgba(255,255,255,0.9)` or `#fff` |
| **Text Secondary** | `rgba(255,255,255,0.6)` |
| **Text Muted** | `rgba(255,255,255,0.45)` |
| **Text Faint** | `rgba(255,255,255,0.4)` |
| **Breadcrumb Links** | `#a1a1a1` |
| **Dropdown Background** | `#252525` |

### Light Mode

| Role | Value |
|------|-------|
| **Page Background** | `#f5f5f4` |
| **Navigation Background** | `#fbfbfa` |
| **Page Header Background** | `#fafafa` |
| **Card/Surface Background** | `#fff` |
| **Card Hover** | `#fafafa` |
| **Border/Divider** | `rgba(0,0,0,0.08)` |
| **Text Primary** | `rgb(55,53,47)` or `#37352f` |
| **Text Secondary** | `rgba(55,53,47,0.6)` |
| **Text Muted** | `rgba(55,53,47,0.45)` |
| **Text Faint** | `rgba(55,53,47,0.4)` |
| **Breadcrumb Links** | `rgba(55,53,47,0.5)` |
| **Dropdown Background** | `#fff` |

---

## 3. Button Styles

### Primary Button (Black - Notion style)
```css
background: #37352f;
color: #fff;
```
Hover: `background: #2f2e2b;`

**Same in both light and dark mode.**

### Secondary Button (Ghost)
```css
/* Dark */
background: transparent;
color: rgba(255,255,255,0.65);

/* Light */
color: rgba(55,53,47,0.65);
```
Hover: slight background `rgba(255,255,255,0.08)` (dark) or `rgba(0,0,0,0.05)` (light)

### Danger Button (Red)
```css
color: #eb5757;
background: transparent;
```
Hover: `background: rgba(235,87,87,0.08);`

---

## 4. Typography

### Font Family
```css
font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
```

### Font Sizes

| Element | Size | Weight |
|---------|------|--------|
| Page title / current page | 12px | 500 |
| Nav items | 12px | normal |
| Buttons | 12px | 500 |
| Section titles (uppercase) | 11px | 500 |
| Stat labels | 11px | normal |
| Stat values | 18px | 600 |
| Card titles | 12px | 500 |
| List items | 12px | normal |
| Badges | 10px | 500 |
| Muted text / subtext | 11px | normal |

### Text Transform
- Section titles: `text-transform: uppercase; letter-spacing: 0.03em;`

---

## 5. Spacing

### Page Layout
```css
/* Main container - offset for nav */
.main-container {
    margin-left: 224px; /* nav width: 14rem = 224px */
    min-height: 100vh;
}
@media (max-width: 767px) {
    .main-container { margin-left: 0; }
}
```

### Content Container
```css
.container {
    width: 100%;
    padding: 20px 16px;
}
@media (min-width: 768px) {
    .container { padding: 24px 32px; }
}
```

### Dashboard Layout
```css
.dash-container {
    width: 100%;
    padding: 20px 16px;
}
@media (min-width: 768px) {
    .dash-container { padding: 24px 32px; }
}

.dash-section {
    margin-bottom: 20px;
}

.page-width {
    width: 100%;
    max-width: 1120px;
    margin: 0 auto;
}
```
Use `.dash-container` for full-width dashboard pages (e.g., index, clients) so sections stretch edge-to-edge while keeping consistent horizontal padding. Wrap grouped content in `.dash-section` blocks for even vertical rhythm, and use `.page-width` when a section should align to the global 1120px max-width (e.g., detailed forms).

### Common Padding
- Card padding: `12px 14px` or `14px 16px`
- List item padding: `8px 12px`
- Button padding: `6px 12px`
- Small button padding: `4px 6px`

### Gaps
- Button groups: `6px` to `8px`
- Card grids: `8px` to `12px`
- Sections: `margin-bottom: 20px`

---

## 6. Border Radius

| Element | Radius |
|---------|--------|
| Buttons | `4px` |
| Cards | `6px` |
| Dropdowns | `6px` |
| Badges | `3px` |
| Nav items | `3px` |
| Avatars | `3px` |

---

## 7. Shadows

### Dark Mode
- Dropdowns: `box-shadow: 0 4px 12px rgba(0,0,0,0.3);`

### Light Mode
- Dropdowns: `box-shadow: 0 4px 12px rgba(0,0,0,0.12);`
- Cards: No shadow, use border only

---

## 8. Page Header Component

Height: `44px`, sticky at top.

```css
.page-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 0 16px;
    height: 44px;
    background: #0f0f0f;
    border-bottom: 1px solid rgba(255,255,255,0.06);
    position: sticky;
    top: 0;
    z-index: 30;
}
@media (min-width: 768px) {
    .page-header {
        padding: 0 32px;
    }
}
[data-theme='light'] .page-header {
    background: transparent;
}
```

### Breadcrumbs
- All links: `#a1a1a1` (dark) / `rgba(55,53,47,0.5)` (light)
- Current page (last item): `#fff` (dark) / `#37352f` (light), weight 500
- Separator: `rgba(255,255,255,0.25)` (dark) / `rgba(0,0,0,0.2)` (light)
- Icon size: `14px`
- Separator icon: `12px`

---

## 9. Cards

```css
.card {
    padding: 14px 16px;
    border-radius: 6px;
    background: rgba(255,255,255,0.03);
    border: 1px solid rgba(255,255,255,0.04);
}
.card:hover {
    background: rgba(255,255,255,0.05);
}
[data-theme='light'] .card {
    background: #fff;
    border: 1px solid rgba(0,0,0,0.08);
}
[data-theme='light'] .card:hover {
    background: #fafafa;
}
```

---

## 10. Lists

```css
.list {
    border-radius: 6px;
    background: rgba(255,255,255,0.02);
    border: 1px solid rgba(255,255,255,0.04);
}
[data-theme='light'] .list {
    background: #fff;
    border: 1px solid rgba(0,0,0,0.08);
}

.list-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 10px 12px;
    border-bottom: 1px solid rgba(255,255,255,0.05);
}
[data-theme='light'] .list-header {
    border-bottom-color: rgba(0,0,0,0.05);
}

.list-item {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 8px 12px;
    border-bottom: 1px solid rgba(255,255,255,0.03);
}
.list-item:last-child { border-bottom: none; }
[data-theme='light'] .list-item {
    border-bottom-color: rgba(0,0,0,0.03);
}
```

---

## 11. Badges

```css
.badge {
    font-size: 10px;
    padding: 2px 6px;
    border-radius: 3px;
    font-weight: 500;
}

/* Success (green) */
.badge-success {
    background: rgba(16,185,129,0.15);
    color: #10b981;
}

/* Info (blue) */
.badge-info {
    background: rgba(59,130,246,0.15);
    color: #3b82f6;
}

/* Default (neutral) */
.badge-default {
    background: rgba(255,255,255,0.08);
    color: rgba(255,255,255,0.5);
}
[data-theme='light'] .badge-default {
    background: rgba(0,0,0,0.06);
    color: rgba(55,53,47,0.6);
}
```

---

## 12. Dropdowns

```css
.dropdown-menu {
    position: absolute;
    right: 0;
    top: 100%;
    margin-top: 4px;
    min-width: 160px;
    background: #252525;
    border-radius: 6px;
    padding: 4px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
    z-index: 100;
}
[data-theme='light'] .dropdown-menu {
    background: #fff;
    box-shadow: 0 4px 12px rgba(0,0,0,0.12);
    border: 1px solid rgba(0,0,0,0.08);
}

.dropdown-item {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 8px 10px;
    border-radius: 4px;
    font-size: 12px;
    color: rgba(255,255,255,0.75);
    cursor: pointer;
}
.dropdown-item:hover {
    background: rgba(255,255,255,0.08);
    color: #fff;
}
[data-theme='light'] .dropdown-item {
    color: rgba(55,53,47,0.75);
}
[data-theme='light'] .dropdown-item:hover {
    background: rgba(0,0,0,0.04);
    color: #37352f;
}
```

---

## 13. Forms (Inputs)

```css
.input {
    width: 100%;
    padding: 8px 12px;
    border-radius: 4px;
    font-size: 13px;
    background: rgba(255,255,255,0.05);
    border: 1px solid rgba(255,255,255,0.08);
    color: rgba(255,255,255,0.9);
}
.input:focus {
    outline: none;
    border-color: rgba(255,255,255,0.2);
}
[data-theme='light'] .input {
    background: #fff;
    border: 1px solid rgba(0,0,0,0.12);
    color: #37352f;
}
[data-theme='light'] .input:focus {
    border-color: rgba(0,0,0,0.25);
}
```

---

## 14. Grid Layouts

### Stats Grid (4 columns)
```css
.stats-grid {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 8px;
}
@media (max-width: 767px) {
    .stats-grid { grid-template-columns: repeat(2, 1fr); }
}
```

### Cards Grid (2-4 columns)
```css
.cards-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 8px;
}
@media (min-width: 1024px) {
    .cards-grid { grid-template-columns: repeat(4, 1fr); }
}
@media (max-width: 639px) {
    .cards-grid { grid-template-columns: 1fr; }
}
```

### 3-Column Grid
```css
.cols-3 {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 12px;
}
@media (max-width: 1023px) {
    .cols-3 { grid-template-columns: 1fr; }
}
```

---

## 15. Icons

- Size in nav/buttons: `14px`
- Size in breadcrumbs: `12px` to `14px`
- Opacity in nav items: `0.55` normal, `0.85` on hover/active
- Always use `stroke="currentColor"` for color inheritance

---

## 16. Transitions

Standard transition for hover effects:
```css
transition: background 0.1s ease;
transition: color 0.1s ease;
```

Chevron rotation:
```css
transition: transform 0.12s ease;
```

---

## 17. Z-Index Scale

| Element | Z-Index |
|---------|---------|
| Dropdowns | 100 |
| Navigation | 40 |
| Page Header | 30 |
| Overlay | 35 |

---

## 18. Page Template

```php
<?php
$currentTheme = $_GET['theme'] ?? 'dark';
?>
<!DOCTYPE html>
<html lang="en" data-theme="<?= h($currentTheme) ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Page Title - Keelance</title>
    <link rel="icon" type="image/png" href="/assets/favicon.png">
    <link rel="stylesheet" href="/assets/styles/app.build.css">
    <style>
        body { margin: 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; }
        body { background: #0f0f0f; color: rgba(255,255,255,0.9); }
        [data-theme='light'] body { background: #f5f5f4; color: #37352f; }
        
        .main-container { margin-left: 224px; min-height: 100vh; }
        @media (max-width: 767px) { .main-container { margin-left: 0; } }
        
        /* Page header styles */
        .page-header { /* ... */ }
        
        /* Content container */
        .container { width: 100%; padding: 24px 32px; }
        @media (max-width: 767px) { .container { padding: 20px 16px; } }
        
        /* Add page-specific styles here */
    </style>
</head>
<body>
    <?php require_once __DIR__ . '/components/navigation.php'; renderNavigationNew('PageName', $currentTheme); ?>

    <div class="main-container">
        <header class="page-header">
            <!-- Breadcrumbs left, Actions right -->
        </header>

        <div class="container">
            <!-- Page content -->
        </div>
    </div>
</body>
</html>
```

---

## 19. Quick Reference - Copy-Paste CSS

### Body Base
```css
body { margin: 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; background: #0f0f0f; color: rgba(255,255,255,0.9); }
[data-theme='light'] body { background: #f5f5f4; color: #37352f; }
```

### Section Title
```css
.section-title { font-size: 11px; font-weight: 500; text-transform: uppercase; letter-spacing: 0.03em; color: rgba(255,255,255,0.45); margin-bottom: 8px; }
[data-theme='light'] .section-title { color: rgba(55,53,47,0.45); }
```

### Primary Button
```css
.btn-primary { display: inline-flex; align-items: center; gap: 6px; padding: 6px 12px; border-radius: 4px; font-size: 12px; font-weight: 500; background: #37352f; color: #fff; border: none; cursor: pointer; }
.btn-primary:hover { background: #2f2e2b; }
```

### Secondary Button
```css
.btn-secondary { display: inline-flex; align-items: center; gap: 6px; padding: 6px 12px; border-radius: 4px; font-size: 12px; font-weight: 500; background: transparent; color: rgba(255,255,255,0.65); border: none; cursor: pointer; }
.btn-secondary:hover { background: rgba(255,255,255,0.08); color: #fff; }
[data-theme='light'] .btn-secondary { color: rgba(55,53,47,0.65); }
[data-theme='light'] .btn-secondary:hover { background: rgba(0,0,0,0.05); color: #37352f; }
```

---

## 20. Files Reference

| File | Purpose |
|------|---------|
| `/components/navigation.php` | Sidebar nav with workspace switcher |
| `/components/headers/page-header.php` | Reusable page header (breadcrumbs + buttons) |
| `/assets/styles/app.css` | Base Tailwind styles |
| `/assets/styles/app.build.css` | Compiled CSS |
| `/index.php` | Dashboard - reference implementation |

---

*Last updated: December 2024*
