<?php

declare(strict_types=1);

require_once __DIR__ . '/../../src/db.php';
require_once __DIR__ . '/../../src/SuperAdmin/SuperAdminAuth.php';
require_once __DIR__ . '/../../src/SuperAdmin/PlatformSettings.php';
require_once __DIR__ . '/navigation.php';

SuperAdminAuth::requirePermission('settings_view');

$admin = SuperAdminAuth::getCurrentAdmin();
$canManage = SuperAdminAuth::hasPermission('settings_manage');

$message = '';
$error = '';

// Handle save
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $canManage) {
    $category = $_POST['category'] ?? '';
    
    if ($category === 'payments') {
        PlatformSettings::set('stripe_enabled', isset($_POST['stripe_enabled']), $admin['id']);
        PlatformSettings::set('stripe_public_key', trim($_POST['stripe_public_key'] ?? ''), $admin['id']);
        PlatformSettings::set('stripe_secret_key', trim($_POST['stripe_secret_key'] ?? ''), $admin['id']);
        PlatformSettings::set('stripe_webhook_secret', trim($_POST['stripe_webhook_secret'] ?? ''), $admin['id']);
        PlatformSettings::set('razorpay_enabled', isset($_POST['razorpay_enabled']), $admin['id']);
        PlatformSettings::set('razorpay_key_id', trim($_POST['razorpay_key_id'] ?? ''), $admin['id']);
        PlatformSettings::set('razorpay_key_secret', trim($_POST['razorpay_key_secret'] ?? ''), $admin['id']);
        PlatformSettings::set('razorpay_webhook_secret', trim($_POST['razorpay_webhook_secret'] ?? ''), $admin['id']);
        $message = 'Payment settings saved successfully.';
    } elseif ($category === 'billing') {
        PlatformSettings::set('trial_enabled', isset($_POST['trial_enabled']), $admin['id']);
        PlatformSettings::set('trial_days', (int)($_POST['trial_days'] ?? 14), $admin['id']);
        PlatformSettings::set('trial_apply_to', $_POST['trial_apply_to'] ?? 'all', $admin['id']);
        PlatformSettings::set('trial_plan_id', (int)($_POST['trial_plan_id'] ?? 1), $admin['id']);
        PlatformSettings::set('discount_enabled', isset($_POST['discount_enabled']), $admin['id']);
        $customRate = (float)($_POST['usd_to_inr_rate'] ?? $currencySettings['usd_to_inr_rate'] ?? 83);
        if ($customRate <= 0) {
            $customRate = 83.0;
        }
        PlatformSettings::set('usd_to_inr_rate', $customRate, $admin['id']);
        $message = 'Billing settings saved successfully.';
    }
    
    SuperAdminAuth::logActivity($admin['id'], 'update_settings', 'settings', null, ['category' => $category]);
    PlatformSettings::clearCache();
}

$paymentSettings = PlatformSettings::getByCategory('payments');
$billingSettings = PlatformSettings::getByCategory('billing');
$currencySettings = PlatformSettings::getCurrencySettings();

// Get plans for trial plan selection
$pdo = getPDO();
$stmt = $pdo->query('SELECT id, name FROM subscription_plans WHERE is_active = 1 ORDER BY sort_order');
$plans = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings · Keelance Admin</title>
    <link rel="icon" type="image/png" href="/assets/favicon.png">
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-slate-100 min-h-screen text-slate-900">
<div class="min-h-screen flex flex-col">
<?php renderAdminNavigation(); ?>

<main class="flex-1">
    <div class="max-w-6xl mx-auto px-4 py-6 space-y-6">
        <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3">
            <div>
                <h1 class="text-2xl font-bold text-slate-900 tracking-tight">Platform Settings</h1>
                <p class="text-slate-500 mt-1 text-sm">Configure payment gateways, trials, and billing behaviour for all workspaces.</p>
            </div>
            <div class="flex items-center gap-2 text-xs text-slate-500">
                <span class="inline-flex items-center gap-1 rounded-full border border-slate-200 bg-white px-2.5 py-1 shadow-sm">
                    <span class="h-1.5 w-1.5 rounded-full bg-blue-500"></span>
                    <span class="font-medium text-slate-700">Platform controls</span>
                </span>
            </div>
        </div>

        <?php if ($message): ?>
            <div class="bg-emerald-50 border border-emerald-200 rounded-lg p-4">
                <p class="text-sm text-emerald-700"><?= htmlspecialchars($message) ?></p>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="bg-red-50 border border-red-200 rounded-lg p-4">
                <p class="text-sm text-red-700"><?= htmlspecialchars($error) ?></p>
            </div>
        <?php endif; ?>

        <div class="space-y-6">
        <!-- Payment Gateway Settings -->
        <div class="bg-white rounded-2xl p-6 border border-slate-200 shadow-sm hover:shadow-md transition-shadow duration-200">
            <div class="flex items-center justify-between mb-4">
                <div>
                    <p class="text-sm font-semibold text-slate-900">Payment gateways</p>
                    <p class="text-xs text-slate-400">Stripe &amp; Razorpay configuration</p>
                </div>
            </div>
            <form method="POST" class="space-y-6">
                <input type="hidden" name="category" value="payments">
                
                <!-- Stripe -->
                <div class="border border-slate-200 rounded-lg p-4">
                    <div class="flex items-center justify-between mb-4">
                        <div class="flex items-center gap-3">
                            <div class="w-10 h-10 bg-indigo-500/20 rounded-lg flex items-center justify-center">
                                <span class="text-indigo-400 font-bold text-sm">S</span>
                            </div>
                            <div>
                                <h3 class="font-medium">Stripe</h3>
                                <p class="text-xs text-slate-500">Accept credit card payments</p>
                            </div>
                        </div>
                        <label class="relative inline-flex items-center cursor-pointer">
                            <input type="checkbox" name="stripe_enabled" class="sr-only peer" <?= ($paymentSettings['stripe_enabled']['value'] ?? false) ? 'checked' : '' ?> <?= !$canManage ? 'disabled' : '' ?>>
                            <div class="w-11 h-6 bg-slate-600 peer-focus:ring-2 peer-focus:ring-blue-500 rounded-full peer peer-checked:after:translate-x-full after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                        </label>
                    </div>
                    <div class="grid md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm text-slate-500 mb-2">Publishable Key</label>
                            <input type="text" name="stripe_public_key" value="<?= htmlspecialchars($paymentSettings['stripe_public_key']['value'] ?? '') ?>"
                                   class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                   placeholder="pk_..." <?= !$canManage ? 'disabled' : '' ?>>
                        </div>
                        <div>
                            <label class="block text-sm text-slate-500 mb-2">Secret Key</label>
                            <input type="password" name="stripe_secret_key" value="<?= htmlspecialchars($paymentSettings['stripe_secret_key']['value'] ?? '') ?>"
                                   class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                   placeholder="sk_..." <?= !$canManage ? 'disabled' : '' ?>>
                        </div>
                        <div class="md:col-span-2">
                            <label class="block text-sm text-slate-500 mb-2">Webhook Secret</label>
                            <input type="password" name="stripe_webhook_secret" value="<?= htmlspecialchars($paymentSettings['stripe_webhook_secret']['value'] ?? '') ?>"
                                   class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                   placeholder="whsec_..." <?= !$canManage ? 'disabled' : '' ?>>
                        </div>
                    </div>
                </div>
                
                <!-- Razorpay -->
                <div class="border border-slate-200 rounded-lg p-4">
                    <div class="flex items-center justify-between mb-4">
                        <div class="flex items-center gap-3">
                            <div class="w-10 h-10 bg-blue-500/20 rounded-lg flex items-center justify-center">
                                <span class="text-blue-400 font-bold text-sm">R</span>
                            </div>
                            <div>
                                <h3 class="font-medium">Razorpay</h3>
                                <p class="text-xs text-slate-500">Accept payments in India</p>
                            </div>
                        </div>
                        <label class="relative inline-flex items-center cursor-pointer">
                            <input type="checkbox" name="razorpay_enabled" class="sr-only peer" <?= ($paymentSettings['razorpay_enabled']['value'] ?? false) ? 'checked' : '' ?> <?= !$canManage ? 'disabled' : '' ?>>
                            <div class="w-11 h-6 bg-slate-600 peer-focus:ring-2 peer-focus:ring-blue-500 rounded-full peer peer-checked:after:translate-x-full after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                        </label>
                    </div>
                    <div class="grid md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm text-slate-500 mb-2">Key ID</label>
                            <input type="text" name="razorpay_key_id" value="<?= htmlspecialchars($paymentSettings['razorpay_key_id']['value'] ?? '') ?>"
                                   class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                   placeholder="rzp_..." <?= !$canManage ? 'disabled' : '' ?>>
                        </div>
                        <div>
                            <label class="block text-sm text-slate-500 mb-2">Key Secret</label>
                            <input type="password" name="razorpay_key_secret" value="<?= htmlspecialchars($paymentSettings['razorpay_key_secret']['value'] ?? '') ?>"
                                   class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                   <?= !$canManage ? 'disabled' : '' ?>>
                        </div>
                        <div class="md:col-span-2">
                            <label class="block text-sm text-slate-500 mb-2">Webhook Secret</label>
                            <input type="password" name="razorpay_webhook_secret" value="<?= htmlspecialchars($paymentSettings['razorpay_webhook_secret']['value'] ?? '') ?>"
                                   class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                   <?= !$canManage ? 'disabled' : '' ?>>
                        </div>
                    </div>
                </div>
                
                <?php if ($canManage): ?>
                    <button type="submit" class="inline-flex items-center gap-1.5 px-5 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg text-sm font-medium transition-colors">
                        Save payment settings
                    </button>
                <?php endif; ?>
            </form>
        </div>

        <!-- Trial & Billing Settings -->
        <div class="bg-white rounded-2xl p-6 border border-slate-200 shadow-sm hover:shadow-md transition-shadow duration-200">
            <div class="flex items-center justify-between mb-4">
                <div>
                    <p class="text-sm font-semibold text-slate-900">Trial &amp; billing</p>
                    <p class="text-xs text-slate-400">Free trial, discounts, and currency rate</p>
                </div>
            </div>
            <form method="POST" class="space-y-4">
                <input type="hidden" name="category" value="billing">
                
                <!-- Info Box -->
                <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 text-sm text-blue-800">
                    <p class="font-medium">How it works:</p>
                    <ul class="mt-2 space-y-1 text-blue-700">
                        <li>• <strong>USD customers</strong> → Stripe (Credit/Debit Cards, ACH)</li>
                        <li>• <strong>INR customers</strong> → Razorpay (UPI, Cards, Net Banking)</li>
                        <li>• Plan prices are set manually in <a href="/admin/plans.php" class="underline">Plans</a> for both currencies</li>
                        <li>• Yearly savings are calculated automatically from plan prices</li>
                    </ul>
                </div>
                
                <div class="border-t border-slate-200 pt-4">
                    <h3 class="font-medium mb-3">Free Trial</h3>
                    <div class="grid md:grid-cols-2 lg:grid-cols-4 gap-4">
                        <div class="flex items-center gap-3">
                            <label class="relative inline-flex items-center cursor-pointer">
                                <input type="checkbox" name="trial_enabled" class="sr-only peer" <?= ($billingSettings['trial_enabled']['value'] ?? true) ? 'checked' : '' ?> <?= !$canManage ? 'disabled' : '' ?>>
                                <div class="w-11 h-6 bg-slate-300 peer-focus:ring-2 peer-focus:ring-blue-500 rounded-full peer peer-checked:after:translate-x-full after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                            </label>
                            <span class="text-sm text-slate-700">Enable Free Trial</span>
                        </div>
                        <div>
                            <label class="block text-sm text-slate-500 mb-2">Trial Duration (days)</label>
                            <input type="number" name="trial_days" value="<?= $billingSettings['trial_days']['value'] ?? 14 ?>"
                                   class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500"
                                   min="1" max="90" <?= !$canManage ? 'disabled' : '' ?>>
                        </div>
                        <div>
                            <label class="block text-sm text-slate-500 mb-2">Apply Trial To</label>
                            <select name="trial_apply_to" class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500" <?= !$canManage ? 'disabled' : '' ?>>
                                <option value="all" <?= ($billingSettings['trial_apply_to']['value'] ?? 'all') === 'all' ? 'selected' : '' ?>>All Plans</option>
                                <option value="specific" <?= ($billingSettings['trial_apply_to']['value'] ?? '') === 'specific' ? 'selected' : '' ?>>Specific Plan Only</option>
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm text-slate-500 mb-2">Trial Plan (if specific)</label>
                            <select name="trial_plan_id" class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500" <?= !$canManage ? 'disabled' : '' ?>>
                                <?php foreach ($plans as $plan): ?>
                                    <option value="<?= $plan['id'] ?>" <?= ($billingSettings['trial_plan_id']['value'] ?? 1) == $plan['id'] ? 'selected' : '' ?>><?= htmlspecialchars($plan['name']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <p class="mt-2 text-xs text-slate-500">New signups will get a free trial. No credit card required during trial.</p>
                </div>
                
                <div class="border-t border-slate-200 pt-4">
                    <h3 class="font-medium mb-3">Discount Codes</h3>
                    <div class="flex items-center gap-3">
                        <label class="relative inline-flex items-center cursor-pointer">
                            <input type="checkbox" name="discount_enabled" class="sr-only peer" <?= ($billingSettings['discount_enabled']['value'] ?? false) ? 'checked' : '' ?> <?= !$canManage ? 'disabled' : '' ?>>
                            <div class="w-11 h-6 bg-slate-300 peer-focus:ring-2 peer-focus:ring-blue-500 rounded-full peer peer-checked:after:translate-x-full after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                        </label>
                        <span class="text-sm text-slate-700">Enable Discount Codes</span>
                    </div>
                    <p class="mt-2 text-xs text-slate-500">Allow customers to apply discount codes at checkout. Manage codes in <a href="/admin/discounts.php" class="text-blue-600 underline">Discounts</a>.</p>
                </div>

                <div class="border-t border-slate-200 pt-4">
                    <h3 class="font-medium mb-3">Currency Conversion</h3>
                    <div class="grid md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm text-slate-500 mb-2">USD → INR Rate</label>
                            <div class="flex items-center gap-3">
                                <span class="text-sm text-slate-500">1 USD =</span>
                                <input type="number" step="0.01" min="1" name="usd_to_inr_rate" value="<?= htmlspecialchars((string)($currencySettings['usd_to_inr_rate'] ?? 83)) ?>" class="flex-1 px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500" <?= !$canManage ? 'disabled' : '' ?>>
                                <span class="text-sm font-medium text-slate-600">INR</span>
                            </div>
                            <p class="mt-2 text-xs text-slate-500">Used for consolidated revenue conversions across dashboards. Update whenever USD↔INR rates change.</p>
                        </div>
                    </div>
                </div>
                
                <?php if ($canManage): ?>
                    <button type="submit" class="inline-flex items-center gap-1.5 px-5 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg text-sm font-medium transition-colors">
                        Save billing settings
                    </button>
                <?php endif; ?>
            </form>
        </div>
        
        <!-- Gateway Status -->
        <div class="bg-white rounded-2xl p-6 border border-slate-200 shadow-sm hover:shadow-md transition-shadow duration-200">
            <div class="flex items-center justify-between mb-4">
                <div>
                    <p class="text-sm font-semibold text-slate-900">Gateway status</p>
                    <p class="text-xs text-slate-400">Live health of Stripe &amp; Razorpay</p>
                </div>
            </div>
            <div class="grid md:grid-cols-2 gap-4">
                <div class="border border-slate-200 rounded-lg p-4">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center gap-3">
                            <span class="text-2xl">💳</span>
                            <div>
                                <h3 class="font-medium">Stripe</h3>
                                <p class="text-xs text-slate-500">USD Payments</p>
                            </div>
                        </div>
                        <?php if (!empty($paymentSettings['stripe_enabled']['value']) && !empty($paymentSettings['stripe_secret_key']['value'])): ?>
                            <span class="px-2 py-1 bg-emerald-100 text-emerald-700 text-xs font-medium rounded-full">Active</span>
                        <?php else: ?>
                            <span class="px-2 py-1 bg-slate-100 text-slate-500 text-xs font-medium rounded-full">Not Configured</span>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="border border-slate-200 rounded-lg p-4">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center gap-3">
                            <span class="text-2xl">🇮🇳</span>
                            <div>
                                <h3 class="font-medium">Razorpay</h3>
                                <p class="text-xs text-slate-500">INR Payments</p>
                            </div>
                        </div>
                        <?php if (!empty($paymentSettings['razorpay_enabled']['value']) && !empty($paymentSettings['razorpay_key_secret']['value'])): ?>
                            <span class="px-2 py-1 bg-emerald-100 text-emerald-700 text-xs font-medium rounded-full">Active</span>
                        <?php else: ?>
                            <span class="px-2 py-1 bg-slate-100 text-slate-500 text-xs font-medium rounded-full">Not Configured</span>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</main>

<?php renderAdminFooter(); ?>
</div>
</body>
</html>
