<?php

declare(strict_types=1);

require_once __DIR__ . '/../../src/db.php';
require_once __DIR__ . '/../../src/SuperAdmin/SuperAdminAuth.php';
require_once __DIR__ . '/../../src/SuperAdmin/WorkspaceManager.php';
require_once __DIR__ . '/../../src/SuperAdmin/PlanManager.php';
require_once __DIR__ . '/navigation.php';

SuperAdminAuth::requirePermission('workspaces_view');

$admin = SuperAdminAuth::getCurrentAdmin();
$canManage = SuperAdminAuth::hasPermission('workspaces_manage');
$canBan = SuperAdminAuth::hasPermission('workspaces_ban');
$canDelete = SuperAdminAuth::hasPermission('workspaces_delete');
$canLoginAs = SuperAdminAuth::hasPermission('workspaces_login');

$message = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $workspaceId = (int)($_POST['workspace_id'] ?? 0);

    if ($action === 'ban' && $canBan && $workspaceId) {
        $reason = trim($_POST['reason'] ?? '');
        if (WorkspaceManager::banWorkspace($workspaceId, $reason)) {
            SuperAdminAuth::logActivity($admin['id'], 'ban_workspace', 'workspace', $workspaceId, ['reason' => $reason]);
            $message = 'Workspace banned successfully.';
        } else {
            $error = 'Failed to ban workspace.';
        }
    } elseif ($action === 'unban' && $canBan && $workspaceId) {
        if (WorkspaceManager::unbanWorkspace($workspaceId)) {
            SuperAdminAuth::logActivity($admin['id'], 'unban_workspace', 'workspace', $workspaceId);
            $message = 'Workspace unbanned successfully.';
        } else {
            $error = 'Failed to unban workspace.';
        }
    } elseif ($action === 'delete' && $canDelete && $workspaceId) {
        if (WorkspaceManager::deleteWorkspace($workspaceId)) {
            SuperAdminAuth::logActivity($admin['id'], 'delete_workspace', 'workspace', $workspaceId);
            $message = 'Workspace deleted successfully.';
        } else {
            $error = 'Failed to delete workspace.';
        }
    } elseif ($action === 'update_subscription' && $canManage && $workspaceId) {
        $planId = (int)($_POST['plan_id'] ?? 0);
        $status = $_POST['subscription_status'] ?? 'active';
        $expiresAt = $_POST['expires_at'] ?: null;
        if (WorkspaceManager::updateSubscription($workspaceId, $planId, $status, $expiresAt)) {
            SuperAdminAuth::logActivity($admin['id'], 'update_subscription', 'workspace', $workspaceId, [
                'plan_id' => $planId, 'status' => $status
            ]);
            $message = 'Subscription updated successfully.';
        } else {
            $error = 'Failed to update subscription.';
        }
    }
}

// Get filters
$page = max(1, (int)($_GET['page'] ?? 1));
$search = trim($_GET['search'] ?? '');
$filters = [];
if ($search) $filters['search'] = $search;
if (isset($_GET['is_banned'])) $filters['is_banned'] = (int)$_GET['is_banned'];
if (!empty($_GET['subscription_status'])) $filters['subscription_status'] = $_GET['subscription_status'];

$workspaceStats = WorkspaceManager::getStats();
$result = WorkspaceManager::getWorkspaces($page, 20, $filters);
$workspaces = $result['workspaces'];
$totalPages = $result['total_pages'];
$total = $result['total'];

$plans = PlanManager::getAllPlans();

function formatWorkspaceDate(?string $dateValue, string $emptyLabel = '—'): string
{
    if (!$dateValue) {
        return '<span class="text-slate-400">' . htmlspecialchars($emptyLabel, ENT_QUOTES) . '</span>';
    }

    $timestamp = strtotime($dateValue);
    if (!$timestamp) {
        return '<span class="text-slate-400">' . htmlspecialchars($emptyLabel, ENT_QUOTES) . '</span>';
    }

    $primary = date('jS M, y', $timestamp);
    return '<span class="text-slate-700 font-medium">' . htmlspecialchars($primary, ENT_QUOTES) . '</span>';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Workspaces · Keelance Admin</title>
    <link rel="icon" type="image/png" href="/assets/favicon.png">
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-slate-100 min-h-screen text-slate-900">
    <div class="min-h-screen flex flex-col">
        <?php renderAdminNavigation(); ?>

        <main class="flex-1">
            <div class="max-w-6xl mx-auto px-4 py-6 space-y-6">
                <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3">
                    <div>
                        <h1 class="text-2xl font-bold text-slate-900 tracking-tight">Workspaces</h1>
                        <p class="text-slate-500 mt-1 text-sm">
                            <?= number_format($total) ?> total workspaces across all plans.
                        </p>
                    </div>
                    <div class="flex items-center gap-2 text-xs text-slate-500">
                        <span class="inline-flex items-center gap-1 rounded-full border border-slate-200 bg-white px-2.5 py-1 shadow-sm">
                            <span class="h-1.5 w-1.5 rounded-full bg-emerald-500"></span>
                            <span class="font-medium text-slate-700">Workspace directory</span>
                        </span>
                        <?php if ($search || isset($_GET['is_banned']) || !empty($_GET['subscription_status'])): ?>
                            <span class="hidden sm:inline text-xs text-slate-400">
                                Filtered view
                            </span>
                        <?php else: ?>
                            <span class="hidden sm:inline text-xs text-slate-400">
                                All workspaces
                            </span>
                        <?php endif; ?>
                    </div>
                </div>

                <?php if ($message): ?>
                    <div class="bg-emerald-50 border border-emerald-200 rounded-lg p-4">
                        <p class="text-sm text-emerald-700"><?= htmlspecialchars($message) ?></p>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="bg-red-50 border border-red-200 rounded-lg p-4">
                        <p class="text-sm text-red-700"><?= htmlspecialchars($error) ?></p>
                    </div>
                <?php endif; ?>

                <!-- Snapshot -->
                <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                    <?php
                        $workspaceCards = [
                            ['label' => 'Total Workspaces', 'value' => number_format($workspaceStats['total_workspaces'] ?? $total), 'chip' => '+'.number_format($workspaceStats['created_this_month'] ?? 0).' new', 'tone' => 'from-slate-50 to-white'],
                            ['label' => 'Active', 'value' => number_format($workspaceStats['active_workspaces'] ?? 0), 'chip' => ($workspaceStats['active_workspaces'] ?? 0) ? round(($workspaceStats['active_workspaces'] / max(1, $workspaceStats['total_workspaces'])) * 100).'%' : '0%', 'tone' => 'from-emerald-50 to-white'],
                            ['label' => 'Trials Running', 'value' => number_format($workspaceStats['by_status']['trial'] ?? 0), 'chip' => 'In free trial', 'tone' => 'from-blue-50 to-white'],
                            ['label' => 'Expiring Soon', 'value' => number_format($workspaceStats['expiring_soon'] ?? 0), 'chip' => 'Next 7 days', 'tone' => 'from-amber-50 to-white'],
                        ];
                    ?>
                    <?php foreach ($workspaceCards as $card): ?>
                        <div class="rounded-2xl border border-slate-200 bg-gradient-to-br <?= $card['tone'] ?> p-4 shadow-sm">
                            <p class="text-xs uppercase tracking-wide text-slate-500 font-semibold"><?= $card['label'] ?></p>
                            <div class="flex items-center justify-between mt-2">
                                <p class="text-2xl font-semibold text-slate-900"><?= $card['value'] ?></p>
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-[11px] font-medium bg-white/70 border border-slate-200 text-slate-600">
                                    <?= htmlspecialchars($card['chip']) ?>
                                </span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Filters -->
                <div class="bg-white border border-slate-200 shadow-sm rounded-2xl p-5 space-y-4">
                    <div class="flex flex-wrap items-center gap-3">
                        <div class="flex items-center gap-2 text-sm text-slate-500">
                            <span class="inline-flex items-center justify-center w-8 h-8 rounded-full bg-slate-100 text-slate-600">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 4a1 1 0 011-1h16a1 1 0 011 1v2H3V4zm0 6h18v2H3v-2zm0 6h18v2a1 1 0 01-1 1H4a1 1 0 01-1-1v-2z" /></svg>
                            </span>
                            <span>Filter workspaces</span>
                        </div>
                        <?php if ($search || isset($_GET['is_banned']) || !empty($_GET['subscription_status'])): ?>
                            <span class="inline-flex items-center gap-1 text-xs font-medium text-blue-700 bg-blue-50 border border-blue-100 px-2.5 py-0.5 rounded-full">
                                Filtered view active
                            </span>
                        <?php endif; ?>
                    </div>
                    <form method="GET" class="flex flex-wrap gap-3">
                        <div class="flex-1 min-w-[220px]">
                            <input type="text" name="search" value="<?= htmlspecialchars($search) ?>"
                                   placeholder="Search by workspace, slug, or owner..."
                                   class="w-full px-4 py-2.5 bg-white border border-slate-300 rounded-lg text-slate-900 placeholder-slate-400 focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div>
                            <select name="subscription_status" class="px-4 py-2.5 bg-white border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="">All Status</option>
                                <option value="active" <?= ($_GET['subscription_status'] ?? '') === 'active' ? 'selected' : '' ?>>Active</option>
                                <option value="trial" <?= ($_GET['subscription_status'] ?? '') === 'trial' ? 'selected' : '' ?>>Trial</option>
                                <option value="past_due" <?= ($_GET['subscription_status'] ?? '') === 'past_due' ? 'selected' : '' ?>>Past Due</option>
                                <option value="canceled" <?= ($_GET['subscription_status'] ?? '') === 'canceled' ? 'selected' : '' ?>>Canceled</option>
                            </select>
                        </div>
                        <div>
                            <select name="is_banned" class="px-4 py-2.5 bg-white border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="">All</option>
                                <option value="0" <?= isset($_GET['is_banned']) && $_GET['is_banned'] === '0' ? 'selected' : '' ?>>Not Banned</option>
                                <option value="1" <?= isset($_GET['is_banned']) && $_GET['is_banned'] === '1' ? 'selected' : '' ?>>Banned</option>
                            </select>
                        </div>
                        <div class="flex items-center gap-2">
                            <button type="submit" class="inline-flex items-center gap-2 px-5 py-2.5 bg-blue-600 hover:bg-blue-700 text-white rounded-lg text-sm font-semibold transition-colors">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" /></svg>
                                Apply
                            </button>
                            <?php if ($search || isset($_GET['is_banned']) || !empty($_GET['subscription_status'])): ?>
                                <a href="/admin/workspaces.php" class="inline-flex items-center gap-2 px-4 py-2.5 rounded-lg border border-slate-200 text-slate-600 text-sm font-medium hover:bg-slate-50">
                                    Reset
                                </a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>

                <!-- Workspaces Table -->
                <div class="bg-white border border-slate-200 shadow-sm rounded-2xl overflow-hidden">
                    <div class="overflow-x-auto">
                        <table class="w-full">
                <thead>
                    <tr class="text-left text-xs font-semibold uppercase tracking-wide text-slate-500 border-b border-slate-200 bg-slate-50">
                        <th class="px-4 py-3">Workspace</th>
                        <th class="px-4 py-3">Owner</th>
                        <th class="px-4 py-3">Plan</th>
                        <th class="px-4 py-3">Billing</th>
                        <th class="px-4 py-3">Status</th>
                        <th class="px-4 py-3">Members</th>
                        <th class="px-4 py-3">Created</th>
                        <th class="px-4 py-3 text-right">Actions</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-slate-100">
                    <?php foreach ($workspaces as $workspace): ?>
                        <tr class="hover:bg-slate-50/60">
                            <td class="px-4 py-4">
                                <div class="space-y-1">
                                    <p class="font-semibold text-slate-900 flex items-center gap-2">
                                        <?= htmlspecialchars($workspace['name']) ?>
                                        <?php if ($workspace['is_banned']): ?>
                                            <span class="inline-flex items-center px-2 py-0.5 rounded-full text-[11px] font-medium bg-red-100 text-red-700">Banned</span>
                                        <?php endif; ?>
                                    </p>
                                    <p class="text-xs text-slate-500">@<?= htmlspecialchars($workspace['username'] ?? '') ?></p>
                                </div>
                            </td>
                            <td class="px-4 py-4">
                                <div class="space-y-1 text-sm">
                                    <p class="font-medium text-slate-800"><?= htmlspecialchars($workspace['owner_name'] ?? 'No owner') ?></p>
                                    <?php if (!empty($workspace['owner_email'])): ?>
                                        <p class="text-xs text-slate-500"><?= htmlspecialchars($workspace['owner_email']) ?></p>
                                    <?php endif; ?>
                                </div>
                            </td>
                            <td class="px-4 py-4">
                                <?php
                                $currency = strtoupper((string)($workspace['preferred_currency'] ?? 'USD'));
                                $cycle = strtolower((string)($workspace['billing_cycle'] ?? 'monthly')) === 'yearly' ? 'Yearly' : 'Monthly';
                                ?>
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-slate-50 text-slate-700 border border-slate-200">
                                    <?= htmlspecialchars($workspace['plan_name'] ?? 'No Plan') ?>
                                </span>
                            </td>
                            <td class="px-4 py-4">
                                <div class="inline-flex items-center gap-2 px-2.5 py-0.5 rounded-full bg-white border border-slate-200 text-slate-600 text-xs">
                                    <span class="inline-flex w-2 h-2 rounded-full <?= $currency === 'USD' ? 'bg-blue-500' : 'bg-amber-500' ?>"></span>
                                    <?= $currency ?> · <?= $cycle ?>
                                </div>
                            </td>
                            <td class="px-4 py-4">
                                <?php if ($workspace['is_banned']): ?>
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-700">
                                        Banned
                                    </span>
                                <?php else: ?>
                                    <?php
                                    $statusColors = [
                                        'active' => 'bg-emerald-100 text-emerald-700',
                                        'trial' => 'bg-blue-100 text-blue-700',
                                        'past_due' => 'bg-amber-100 text-amber-700',
                                        'canceled' => 'bg-red-100 text-red-700',
                                    ];
                                    $status = $workspace['subscription_status'] ?? 'none';
                                    ?>
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?= $statusColors[$status] ?? 'bg-slate-100 text-slate-700' ?>">
                                        <?= ucfirst($status) ?>
                                    </span>
                                <?php endif; ?>
                            </td>
                            <td class="px-4 py-4 text-slate-700">
                                <?= $workspace['member_count'] ?>
                            </td>
                            <td class="px-4 py-4 text-slate-500 text-sm">
                                <?= formatWorkspaceDate($workspace['created_at']); ?>
                            </td>
                            <td class="px-4 py-4">
                                <div class="flex items-center justify-end gap-2">
                                    <?php if ($canManage): ?>
                                        <button onclick="showSubscriptionModal(<?= $workspace['id'] ?>, '<?= htmlspecialchars($workspace['name'], ENT_QUOTES) ?>', <?= $workspace['plan_id'] ?? 'null' ?>, '<?= $workspace['subscription_status'] ?? '' ?>')"
                                                class="w-8 h-8 inline-flex items-center justify-center rounded-full border border-blue-200 text-blue-600 bg-white hover:bg-blue-50 transition-colors" title="Edit plan">
                                            <span class="sr-only">Edit plan</span>
                                            <svg class="w-4 h-4" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                <path d="M12 20h9" />
                                                <path d="M16.5 3.5a2.121 2.121 0 0 1 3 3L7 19l-4 1 1-4 12.5-12.5z" />
                                            </svg>
                                        </button>
                                    <?php endif; ?>
                                    
                                    <?php if ($canBan): ?>
                                        <?php if ($workspace['is_banned']): ?>
                                            <form method="POST" class="inline">
                                                <input type="hidden" name="action" value="unban">
                                                <input type="hidden" name="workspace_id" value="<?= $workspace['id'] ?>">
                                                <button type="submit" class="w-8 h-8 inline-flex items-center justify-center rounded-full border border-emerald-200 text-emerald-600 bg-white hover:bg-emerald-50 transition-colors" title="Unban workspace">
                                                    <span class="sr-only">Unban workspace</span>
                                                    <svg class="w-4 h-4" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                        <path d="M9 11l3 3L22 4" />
                                                        <path d="M21 12v7a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11" />
                                                    </svg>
                                                </button>
                                            </form>
                                        <?php else: ?>
                                            <button onclick="showBanModal(<?= $workspace['id'] ?>, '<?= htmlspecialchars($workspace['name'], ENT_QUOTES) ?>')"
                                                    class="w-8 h-8 inline-flex items-center justify-center rounded-full border border-amber-200 text-amber-600 bg-white hover:bg-amber-50 transition-colors" title="Ban workspace">
                                                <span class="sr-only">Ban workspace</span>
                                                <svg class="w-4 h-4" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                    <circle cx="12" cy="12" r="10" />
                                                    <line x1="4" y1="4" x2="20" y2="20" />
                                                </svg>
                                            </button>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                    
                                    <?php if ($canDelete): ?>
                                        <form method="POST" class="inline" onsubmit="return confirm('Are you sure you want to delete this workspace? All data will be lost. This cannot be undone.')">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="workspace_id" value="<?= $workspace['id'] ?>">
                                            <button type="submit" class="w-8 h-8 inline-flex items-center justify-center rounded-full border border-red-200 text-red-600 bg-white hover:bg-red-50 transition-colors" title="Delete workspace">
                                                <span class="sr-only">Delete workspace</span>
                                                <svg class="w-4 h-4" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                    <polyline points="3 6 5 6 21 6" />
                                                    <path d="M19 6l-1 14a2 2 0 0 1-2 2H8a2 2 0 0 1-2-2L5 6" />
                                                    <path d="M10 11v6" />
                                                    <path d="M14 11v6" />
                                                    <path d="M9 6V4a1 1 0 0 1 1-1h4a1 1 0 0 1 1 1v2" />
                                                </svg>
                                            </button>
                                        </form>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

                        <?php if (empty($workspaces)): ?>
                            <div class="text-center py-12">
                                <p class="text-slate-500">No workspaces found</p>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <div class="flex justify-center gap-2 mt-6">
                            <?php for ($i = 1; $i <= min($totalPages, 10); $i++): ?>
                                <a href="?page=<?= $i ?>&search=<?= urlencode($search) ?><?= isset($_GET['subscription_status']) ? '&subscription_status=' . urlencode((string) $_GET['subscription_status']) : '' ?><?= isset($_GET['is_banned']) ? '&is_banned=' . urlencode((string) $_GET['is_banned']) : '' ?>"
                                   class="px-4 py-2 rounded-full text-sm <?= $i === $page
                                        ? 'bg-blue-600 border border-blue-600 text-white'
                                        : 'bg-white border border-slate-200 text-slate-600 hover:bg-slate-50' ?>">
                                    <?= $i ?>
                                </a>
                            <?php endfor; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>

<!-- Ban Modal -->
<div id="banModal" class="fixed inset-0 bg-black/50 hidden items-center justify-center z-50">
    <div class="bg-white border border-slate-200 shadow-sm rounded-xl p-6 w-full max-w-md mx-4">
        <h3 class="text-lg font-semibold mb-4">Ban Workspace</h3>
        <form method="POST">
            <input type="hidden" name="action" value="ban">
            <input type="hidden" name="workspace_id" id="banWorkspaceId">
            <p class="text-slate-500 mb-4">Are you sure you want to ban <span id="banWorkspaceName" class="text-slate-900 font-medium"></span>?</p>
            <div class="mb-4">
                <label class="block text-sm text-slate-500 mb-2">Reason (optional)</label>
                <textarea name="reason" rows="3" class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 placeholder-slate-400 focus:outline-none focus:ring-2 focus:ring-blue-500"></textarea>
            </div>
            <div class="flex gap-3">
                <button type="button" onclick="closeBanModal()" class="flex-1 px-4 py-2 bg-slate-50 hover:bg-slate-600 rounded-lg transition-colors">
                    Cancel
                </button>
                <button type="submit" class="flex-1 px-4 py-2 bg-red-600 hover:bg-red-700 rounded-lg transition-colors">
                    Ban Workspace
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Subscription Modal -->
<div id="subscriptionModal" class="fixed inset-0 bg-black/50 hidden items-center justify-center z-50">
    <div class="bg-white border border-slate-200 shadow-sm rounded-xl p-6 w-full max-w-md mx-4">
        <h3 class="text-lg font-semibold mb-4">Update Subscription</h3>
        <form method="POST">
            <input type="hidden" name="action" value="update_subscription">
            <input type="hidden" name="workspace_id" id="subWorkspaceId">
            <p class="text-slate-500 mb-4">Updating subscription for <span id="subWorkspaceName" class="text-slate-900 font-medium"></span></p>
            
            <div class="mb-4">
                <label class="block text-sm text-slate-500 mb-2">Plan</label>
                <select name="plan_id" id="subPlanId" class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <option value="">No Plan</option>
                    <?php foreach ($plans as $plan): ?>
                        <option value="<?= $plan['id'] ?>"><?= htmlspecialchars($plan['name']) ?> - $<?= $plan['price'] ?>/mo</option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="mb-4">
                <label class="block text-sm text-slate-500 mb-2">Status</label>
                <select name="subscription_status" id="subStatus" class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <option value="trial">Trial</option>
                    <option value="active">Active</option>
                    <option value="past_due">Past Due</option>
                    <option value="canceled">Canceled</option>
                </select>
            </div>
            
            <div class="mb-4">
                <label class="block text-sm text-slate-500 mb-2">Expires At</label>
                <input type="datetime-local" name="expires_at" id="subExpiresAt" 
                       class="w-full px-4 py-2 bg-slate-50 border border-slate-300 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500">
            </div>
            
            <div class="flex gap-3">
                <button type="button" onclick="closeSubscriptionModal()" class="flex-1 px-4 py-2 bg-slate-50 hover:bg-slate-600 rounded-lg transition-colors">
                    Cancel
                </button>
                <button type="submit" class="flex-1 px-4 py-2 bg-blue-600 hover:bg-blue-700 rounded-lg transition-colors">
                    Update
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function showBanModal(workspaceId, workspaceName) {
    document.getElementById('banWorkspaceId').value = workspaceId;
    document.getElementById('banWorkspaceName').textContent = workspaceName;
    document.getElementById('banModal').classList.remove('hidden');
    document.getElementById('banModal').classList.add('flex');
}

function closeBanModal() {
    document.getElementById('banModal').classList.add('hidden');
    document.getElementById('banModal').classList.remove('flex');
}

function showSubscriptionModal(workspaceId, workspaceName, planId, status) {
    document.getElementById('subWorkspaceId').value = workspaceId;
    document.getElementById('subWorkspaceName').textContent = workspaceName;
    document.getElementById('subPlanId').value = planId || '';
    document.getElementById('subStatus').value = status || 'trial';
    document.getElementById('subscriptionModal').classList.remove('hidden');
    document.getElementById('subscriptionModal').classList.add('flex');
}

function closeSubscriptionModal() {
    document.getElementById('subscriptionModal').classList.add('hidden');
    document.getElementById('subscriptionModal').classList.remove('flex');
}
</script>
<?php renderAdminFooter(); ?>
</body>
</html>
