# Keelance UI Components

Reusable PHP components for maintaining consistency across all pages.

## Component Structure

```
/public/components/
├── headers/              # Page header variants
│   ├── page-header.php           # Standard header with breadcrumbs + CTAs
│   ├── page-header-simple.php    # Simple header (Profile/Settings)
│   └── page-header-tabs.php      # Header with tab navigation
├── hero/                 # Hero section variants
│   ├── page-hero.php             # Standard hero with title + description
│   └── page-hero-stats.php       # Hero with stat chips
└── banners/              # Status and notification banners
    ├── status-banner.php         # Success/error/warning/info banners
    └── permission-banner.php     # Permission denied message
```

## Usage Examples

### 1. Standard Page Header (Templates, Modules, Clients)

```php
<?php require_once __DIR__ . '/components/headers/page-header.php'; ?>

<?php
renderPageHeader(
    breadcrumbs: [
        ['label' => 'Workspace Name', 'url' => '/index.php', 'icon' => 'M3 7h18M3 12h18M3 17h18'],
        ['label' => 'Proposals', 'url' => '/proposals.php', 'icon' => '']
    ],
    currentPage: 'Templates',
    actions: [
        [
            'label' => 'Create proposal',
            'url' => '/generate.php',
            'type' => 'secondary',
            'icon' => 'M12 4v16m8-8H4',
            'append_workspace' => false
        ],
        [
            'label' => 'New template',
            'url' => '/template_create.php',
            'type' => 'primary',
            'icon' => 'M12 4v16m8-8H4',
            'append_workspace' => true
        ]
    ],
    workspaceQuery: $workspaceQuery
);
?>
```

### 2. Simple Page Header (Profile, Settings without workspace context)

```php
<?php require_once __DIR__ . '/components/headers/page-header-simple.php'; ?>

<?php
renderSimplePageHeader(
    breadcrumbs: [
        ['label' => 'Account', 'icon' => 'M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z']
    ],
    currentPage: 'Profile',
    actions: [
        ['label' => 'Billing', 'url' => '/billing.php', 'type' => 'secondary', 'icon' => '...'],
        ['label' => 'Settings', 'url' => '/settings.php', 'type' => 'secondary', 'icon' => '...']
    ],
    workspaceQuery: $workspaceQuery
);
?>
```

### 3. Page Header with Tabs (Settings, Members, Activity)

```php
<?php require_once __DIR__ . '/components/headers/page-header-tabs.php'; ?>

<?php
renderPageHeaderWithTabs(
    breadcrumbs: [
        ['label' => 'Workspace Settings', 'url' => '/settings.php', 'icon' => '...']
    ],
    currentPage: 'Settings',
    actions: [
        ['label' => 'Profile', 'url' => '/profile.php', 'type' => 'secondary', 'icon' => '...'],
        ['label' => 'Billing', 'url' => '/billing.php', 'type' => 'primary', 'icon' => '...']
    ],
    tabs: [
        ['label' => 'Workspace settings', 'url' => '/settings.php', 'active' => true],
        ['label' => 'Team', 'url' => '/members.php', 'active' => false],
        ['label' => 'Activity log', 'url' => '/activity.php', 'active' => false]
    ],
    workspaceQuery: $workspaceQuery
);
?>
```

### 4. Standard Page Hero

```php
<?php require_once __DIR__ . '/components/hero/page-hero.php'; ?>

<?php
renderPageHero(
    label: 'DASHBOARD',
    title: 'Overview for ' . htmlspecialchars($workspaceName),
    description: 'Stay on top of clients, projects and billing from a single view.',
    extras: [] // Optional: array of HTML strings for chips/badges
);
?>
```

### 5. Page Hero with Stats (Templates/Modules)

```php
<?php require_once __DIR__ . '/components/hero/page-hero-stats.php'; ?>

<?php
renderPageHeroWithStats(
    label: 'LIBRARY',
    title: 'Proposal Templates',
    description: 'Reusable templates for faster proposal generation.',
    stats: [
        ['label' => 'Total', 'value' => '24', 'type' => 'primary'],
        ['label' => 'This month', 'value' => '3', 'type' => 'neutral']
    ]
);
?>
```

### 6. Status Banners

```php
<?php require_once __DIR__ . '/components/banners/status-banner.php'; ?>

<?php if ($successMessage): ?>
    <?php renderStatusBanner('success', $successMessage); ?>
<?php endif; ?>

<?php if (!empty($errors)): ?>
    <?php renderStatusBanner('error', 'Please fix the following errors:', $errors); ?>
<?php endif; ?>
```

### 7. Permission Denied Banner

```php
<?php require_once __DIR__ . '/components/banners/permission-banner.php'; ?>

<?php if (!$hasAccess): ?>
    <?php 
    renderPermissionBanner(
        title: 'Access Restricted',
        message: 'You don\'t have permission to view this page for ',
        workspaceName: $workspaceName,
        backUrl: '/index.php' . $workspaceQuery,
        backLabel: 'Back to Dashboard'
    );
    ?>
<?php endif; ?>
```

## Component Parameters

### Header Components

All header components accept:
- `breadcrumbs`: Array of breadcrumb items with `label`, `url`, and optional `icon`
- `currentPage`: String for the current page title (shown in white)
- `actions`: Array of action buttons with `label`, `url`, `type` (primary/secondary), `icon`, and optional `append_workspace`
- `workspaceQuery`: Workspace query string (e.g., `?workspace=slug`)

### Hero Components

- `label`: Uppercase label text
- `title`: Main heading
- `description`: Subtitle text
- `stats` or `extras`: Additional content (stats chips or custom HTML)

### Banner Components

- `type`: 'success', 'error', 'warning', 'info'
- `message`: Main message text
- `details`: Optional array of detail messages (for errors)
- `workspaceName`: Optional workspace name
- `backUrl`: Optional back button URL
- `backLabel`: Optional back button label

## Design Consistency

All components use:
- Global CSS classes from `/public/assets/styles/app.css`
- Consistent spacing, colors, and typography
- Emerald accent color (`#10b981`)
- Breadcrumb color `#A1A1A1` for non-active items
- Workspace-aware URLs when needed

## Migration Guide

To migrate an existing page:

1. **Include the component file:**
   ```php
   <?php require_once __DIR__ . '/components/headers/page-header.php'; ?>
   ```

2. **Replace header markup with component call:**
   ```php
   <?php renderPageHeader(...); ?>
   ```

3. **Remove inline styles** - use global CSS classes instead

4. **Test breadcrumbs** - ensure workspace context is preserved

5. **Verify CTAs** - check button types and workspace query appending
