<?php
/**
 * Page Header with Tabs Component (Settings/Members/Activity style)
 * 
 * Displays header with breadcrumbs, actions, and tab navigation below
 * 
 * @param array $breadcrumbs Array of breadcrumb items
 * @param string $currentPage Current page title
 * @param array $actions Array of action buttons
 * @param array $tabs Array of tabs [['label' => 'Settings', 'url' => '/settings.php', 'active' => true], ...]
 * @param string|null $workspaceQuery Workspace query string
 */

function renderPageHeaderWithTabs(array $breadcrumbs, string $currentPage, array $actions, array $tabs, ?string $workspaceQuery = ''): void {
?>
<!-- Header -->
<header class="sticky top-0 z-30 flex items-center justify-between px-3 md:px-6 h-12 border-b border-neutral-800 bg-neutral-950/95 backdrop-blur supports-[backdrop-filter]:backdrop-blur-md">
    <div class="flex items-center gap-3 text-sm text-neutral-400">
        <button id="mobile-menu-btn" class="md:hidden inline-flex items-center justify-center p-2 rounded-lg hover:text-neutral-100 hover:bg-neutral-800 transition">
            <svg class="h-5 w-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
            </svg>
        </button>
        
        <nav class="flex items-center gap-2 text-xs">
            <?php foreach ($breadcrumbs as $crumb): ?>
                <a href="<?= htmlspecialchars($crumb['url'] . $workspaceQuery) ?>" class="inline-flex items-center gap-1 text-[#A1A1A1] hover:text-neutral-100 transition truncate max-w-[140px] md:max-w-xs">
                    <?php if (!empty($crumb['icon'])): ?>
                        <svg class="h-3.5 w-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="<?= $crumb['icon'] ?>"></path>
                        </svg>
                    <?php endif; ?>
                    <?= htmlspecialchars($crumb['label']) ?>
                </a>
                <svg class="h-3 w-3 text-neutral-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            <?php endforeach; ?>
            <span class="text-neutral-50 font-semibold"><?= htmlspecialchars($currentPage) ?></span>
        </nav>
    </div>
    
    <div class="flex items-center gap-2 text-xs">
        <?php foreach ($actions as $action): ?>
            <?php $buttonClass = ($action['type'] ?? 'secondary') === 'primary' ? 'btn-emerald' : 'btn-secondary'; ?>
            <a href="<?= htmlspecialchars($action['url'] . $workspaceQuery) ?>" class="<?= $buttonClass ?> inline-flex items-center gap-1 px-3 py-1">
                <?php if (!empty($action['icon'])): ?>
                    <svg class="h-3.5 w-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="<?= $action['icon'] ?>"></path>
                    </svg>
                <?php endif; ?>
                <?= htmlspecialchars($action['label']) ?>
            </a>
        <?php endforeach; ?>
    </div>
</header>

<!-- Tab Navigation -->
<div class="border-b border-neutral-800 bg-neutral-950">
    <div class="max-w-7xl mx-auto px-3 md:px-6">
        <nav class="-mb-px flex space-x-4 text-xs">
            <?php foreach ($tabs as $tab): ?>
                <a href="<?= htmlspecialchars($tab['url'] . $workspaceQuery) ?>" 
                   class="px-3 pb-2 border-b-2 font-medium <?= $tab['active'] ? 'border-neutral-50 text-neutral-50' : 'border-transparent text-neutral-500 hover:text-neutral-200 hover:border-neutral-600' ?>">
                    <?= htmlspecialchars($tab['label']) ?>
                </a>
            <?php endforeach; ?>
        </nav>
    </div>
</div>
<?php
}
