<?php
/**
 * Standard Page Header Component - Notion-inspired
 * 
 * Displays a compact header with breadcrumbs and action buttons
 * 
 * @param array $breadcrumbs Array of breadcrumb items [['label' => 'Home', 'url' => '/index.php', 'icon' => 'svg_path'], ...]
 * @param string $currentPage Current page title (last breadcrumb, shown in white)
 * @param array $actions Array of action buttons [['label' => 'New', 'url' => '/create', 'type' => 'primary|secondary', 'icon' => 'svg_path'], ...]
 * @param string|null $workspaceQuery Workspace query string (e.g., '?workspace=slug')
 */

function renderPageHeader(array $breadcrumbs, string $currentPage, array $actions = [], ?string $workspaceQuery = ''): void {
?>
<style>
    .page-header {
        position: sticky;
        top: 0;
        z-index: 30;
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 0 16px;
        height: 44px;
        background: #0f0f0f;
        border-bottom: 1px solid rgba(255, 255, 255, 0.06);
        border-left: 1px solid rgba(255, 255, 255, 0.06);
        backdrop-filter: blur(8px);
        -webkit-backdrop-filter: blur(8px);
    }
    
    @media (min-width: 768px) {
        .page-header {
            padding: 0 32px;
        }
    }
    
    [data-theme='light'] .page-header {
        background: #fafafa;
        border-bottom: 1px solid rgba(0, 0, 0, 0.08);
        border-left: 1px solid rgba(0, 0, 0, 0.08);
        backdrop-filter: blur(8px);
        -webkit-backdrop-filter: blur(8px);
    }
    
    .header-breadcrumbs {
        display: flex;
        align-items: center;
        gap: 4px;
        font-size: 12px;
        min-width: 0;
    }
    
    .header-breadcrumb-link {
        display: inline-flex;
        align-items: center;
        gap: 4px;
        color: #a1a1a1;
        transition: color 0.1s ease;
    }
    
    .header-breadcrumb-link:hover {
        color: rgba(255, 255, 255, 0.9);
    }
    
    [data-theme='light'] .header-breadcrumb-link {
        color: rgba(55, 53, 47, 0.5);
    }
    
    [data-theme='light'] .header-breadcrumb-link:hover {
        color: rgb(55, 53, 47);
    }
    
    .header-breadcrumb-link svg {
        width: 12px;
        height: 12px;
        flex-shrink: 0;
        opacity: 0.6;
    }
    
    .header-breadcrumb-separator {
        color: rgba(255, 255, 255, 0.2);
        margin: 0 2px;
    }
    
    .header-breadcrumb-separator svg {
        width: 10px;
        height: 10px;
    }
    
    [data-theme='light'] .header-breadcrumb-separator {
        color: rgba(0, 0, 0, 0.2);
    }
    
    .header-current-page {
        font-weight: 500;
        color: rgba(255, 255, 255, 0.9);
    }
    
    [data-theme='light'] .header-current-page {
        color: rgb(55, 53, 47);
    }
    
    .header-actions {
        display: flex;
        align-items: center;
        gap: 6px;
    }
    
    .header-btn {
        display: inline-flex;
        align-items: center;
        gap: 4px;
        padding: 5px 10px;
        border-radius: 4px;
        font-size: 12px;
        font-weight: 500;
        transition: background 0.1s ease;
        cursor: pointer;
        border: none;
        text-decoration: none;
    }
    
    .header-btn svg {
        width: 14px;
        height: 14px;
        flex-shrink: 0;
    }
    
    .header-btn-primary {
        background: #37352f;
        color: #fff;
    }
    
    .header-btn-primary:hover {
        background: #2f2e2b;
    }
    
    [data-theme='light'] .header-btn-primary {
        background: #37352f;
        color: #fff;
    }
    
    .header-btn-secondary {
        background: transparent;
        color: rgba(255, 255, 255, 0.6);
    }
    
    .header-btn-secondary:hover {
        background: rgba(255, 255, 255, 0.06);
        color: rgba(255, 255, 255, 0.9);
    }
    
    [data-theme='light'] .header-btn-secondary {
        color: rgba(55, 53, 47, 0.65);
    }
    
    [data-theme='light'] .header-btn-secondary:hover {
        background: rgba(0, 0, 0, 0.04);
        color: rgb(55, 53, 47);
    }
    
    /* Dropdown */
    .header-dropdown {
        position: relative;
    }
    
    .header-dropdown-menu {
        display: none;
        position: absolute;
        right: 0;
        top: 100%;
        margin-top: 4px;
        min-width: 160px;
        background: #252525;
        border-radius: 6px;
        padding: 4px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        z-index: 100;
    }
    
    .header-dropdown-menu.show {
        display: block;
    }
    
    [data-theme='light'] .header-dropdown-menu {
        background: #fff;
        box-shadow: 0 4px 12px rgba(0,0,0,0.12);
        border: 1px solid rgba(0,0,0,0.08);
    }
    
    .header-dropdown-item {
        display: flex;
        align-items: center;
        gap: 8px;
        width: 100%;
        padding: 6px 10px;
        border-radius: 4px;
        font-size: 12px;
        color: rgba(255, 255, 255, 0.7);
        background: transparent;
        border: none;
        cursor: pointer;
        text-decoration: none;
        transition: background 0.1s ease;
    }
    
    .header-dropdown-item:hover {
        background: rgba(255, 255, 255, 0.08);
        color: #fff;
    }
    
    .header-dropdown-item svg {
        width: 14px;
        height: 14px;
        opacity: 0.6;
    }
    
    [data-theme='light'] .header-dropdown-item {
        color: rgba(55, 53, 47, 0.7);
    }
    
    [data-theme='light'] .header-dropdown-item:hover {
        background: rgba(0, 0, 0, 0.04);
        color: rgb(55, 53, 47);
    }
    
    .header-mobile-menu {
        display: none;
        align-items: center;
        justify-content: center;
        padding: 4px;
        border-radius: 3px;
        color: rgba(255, 255, 255, 0.5);
        background: transparent;
        border: none;
        cursor: pointer;
        margin-right: 8px;
    }
    
    .header-mobile-menu:hover {
        background: rgba(255, 255, 255, 0.05);
        color: rgba(255, 255, 255, 0.8);
    }
    
    .header-mobile-menu svg {
        width: 16px;
        height: 16px;
    }
    
    [data-theme='light'] .header-mobile-menu {
        color: rgba(55, 53, 47, 0.5);
    }
    
    [data-theme='light'] .header-mobile-menu:hover {
        background: rgba(0, 0, 0, 0.03);
        color: rgb(55, 53, 47);
    }
    
    @media (max-width: 767px) {
        .header-mobile-menu {
            display: inline-flex;
        }
        
        .page-header {
            padding: 0 8px;
        }
        
        .header-actions {
            gap: 4px;
        }
        
        .header-btn {
            padding: 5px 8px;
        }
    }
</style>

<header class="page-header">
    <div class="header-breadcrumbs">
        <!-- Mobile Menu Button -->
        <button id="mobile-menu-btn" class="header-mobile-menu" onclick="if(window.openAppSidebar) openAppSidebar();">
            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
            </svg>
        </button>
        
        <!-- Breadcrumbs -->
        <?php foreach ($breadcrumbs as $crumb): ?>
            <?php
                $crumbUrl = $crumb['url'] ?? '#';
                if (!empty($crumb['append_workspace']) && $workspaceQuery && !str_contains($crumbUrl, 'workspace=')) {
                    $separator = str_contains($crumbUrl, '?') ? '&' : '?';
                    $crumbUrl .= $separator . ltrim($workspaceQuery, '?');
                }
            ?>
            <a href="<?= htmlspecialchars($crumbUrl) ?>" class="header-breadcrumb-link">
                <?php if (!empty($crumb['icon'])): ?>
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="<?= $crumb['icon'] ?>"></path>
                    </svg>
                <?php endif; ?>
                <span><?= htmlspecialchars($crumb['label']) ?></span>
            </a>
            <span class="header-breadcrumb-separator">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                </svg>
            </span>
        <?php endforeach; ?>
        <span class="header-current-page"><?= htmlspecialchars($currentPage) ?></span>
    </div>
    
    <!-- Action Buttons -->
    <div class="header-actions">
        <?php foreach ($actions as $action): ?>
            <?php 
                $buttonClass = ($action['type'] ?? 'secondary') === 'primary' ? 'header-btn header-btn-primary' : 'header-btn header-btn-secondary';
                $url = $action['url'] ?? '#';
                if (!empty($action['append_workspace']) && $workspaceQuery) {
                    $separator = str_contains($url, '?') ? '&' : '?';
                    $url .= $separator . ltrim($workspaceQuery, '?');
                }
            ?>
            <?php if (!empty($action['dropdown'])): ?>
                <!-- Dropdown Button -->
                <div class="header-dropdown">
                    <button type="button" class="<?= $buttonClass ?>" onclick="this.parentElement.querySelector('.header-dropdown-menu').classList.toggle('show')">
                        <?php if (!empty($action['icon'])): ?>
                            <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="<?= $action['icon'] ?>"></path>
                            </svg>
                        <?php endif; ?>
                        <span><?= htmlspecialchars($action['label']) ?></span>
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24" style="width:10px;height:10px;opacity:0.6;margin-left:2px;">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                        </svg>
                    </button>
                    <div class="header-dropdown-menu">
                        <?php foreach ($action['dropdown'] as $dropdownItem): ?>
                            <?php 
                                $dropdownUrl = $dropdownItem['url'] ?? '#';
                                if (!empty($dropdownItem['append_workspace']) && $workspaceQuery) {
                                    $separator = str_contains($dropdownUrl, '?') ? '&' : '?';
                                    $dropdownUrl .= $separator . ltrim($workspaceQuery, '?');
                                }
                            ?>
                            <a href="<?= htmlspecialchars($dropdownUrl) ?>" class="header-dropdown-item">
                                <?php if (!empty($dropdownItem['icon'])): ?>
                                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="<?= $dropdownItem['icon'] ?>"></path>
                                    </svg>
                                <?php endif; ?>
                                <span><?= htmlspecialchars($dropdownItem['label']) ?></span>
                            </a>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php elseif (!empty($action['id'])): ?>
                <button id="<?= htmlspecialchars($action['id']) ?>" type="button" class="<?= $buttonClass ?>">
                    <?php if (!empty($action['icon'])): ?>
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="<?= $action['icon'] ?>"></path>
                        </svg>
                    <?php endif; ?>
                    <span><?= htmlspecialchars($action['label']) ?></span>
                </button>
            <?php else: ?>
                <a href="<?= htmlspecialchars($url) ?>" class="<?= $buttonClass ?>">
                    <?php if (!empty($action['icon'])): ?>
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="<?= $action['icon'] ?>"></path>
                        </svg>
                    <?php endif; ?>
                    <span><?= htmlspecialchars($action['label']) ?></span>
                </a>
            <?php endif; ?>
        <?php endforeach; ?>
    </div>
</header>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Close dropdowns when clicking outside
    document.addEventListener('click', function(e) {
        const dropdowns = document.querySelectorAll('.header-dropdown');
        dropdowns.forEach(function(dropdown) {
            if (!dropdown.contains(e.target)) {
                const menu = dropdown.querySelector('.header-dropdown-menu');
                if (menu) {
                    menu.classList.remove('show');
                }
            }
        });
    });
});
</script>

<?php
}
