<?php

declare(strict_types=1);

require_once __DIR__ . '/../src/secure_workspace.php';
require_once __DIR__ . '/../src/Auth/AuthManager.php';
require_once __DIR__ . '/../src/DatabaseHelper.php';
require_once __DIR__ . '/../src/Permissions/PermissionManager.php';

/**
 * Recursively delete a directory and its contents.
 */
function deleteWorkspaceDirectory(string $directory): void
{
    if (!is_dir($directory)) {
        return;
    }
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($directory, FilesystemIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );
    foreach ($iterator as $fileInfo) {
        $pathName = $fileInfo->getPathname();
        if ($fileInfo->isDir()) {
            @rmdir($pathName);
        } else {
            @unlink($pathName);
        }
    }
    @rmdir($directory);
}

// Get company from session first (most reliable), then URL parameter
$company = null;
$workspaceParam = $_GET['workspace'] ?? null;
$pdo = getPDO();

// Priority 1: Use session company if available
if (isset($_SESSION['current_company_id'])) {
    $stmt = $pdo->prepare("SELECT * FROM companies WHERE id = ?");
    $stmt->execute([$_SESSION['current_company_id']]);
    $company = $stmt->fetch();
}

// Priority 2: Use URL parameter if session company not found
if (!$company && $workspaceParam) {
    if (is_string($workspaceParam) && !is_numeric($workspaceParam)) {
        // Search by username
        $stmt = $pdo->prepare("SELECT * FROM companies WHERE username = ?");
        $stmt->execute([$workspaceParam]);
        $company = $stmt->fetch();
    } else {
        // Search by ID
        $stmt = $pdo->prepare("SELECT * FROM companies WHERE id = ?");
        $stmt->execute([(int)$workspaceParam]);
        $company = $stmt->fetch();
    }
}

// Priority 3: Fallback to user's first company
if (!$company) {
    $user = AuthManager::getCurrentUser();
    $stmt = $pdo->prepare("SELECT c.* FROM companies c 
                          JOIN user_company_access uca ON c.id = uca.company_id 
                          WHERE uca.user_id = ? 
                          ORDER BY c.id LIMIT 1");
    $stmt->execute([$user['id']]);
    $company = $stmt->fetch();
}

// Update session if we found a different company
if ($company && (!isset($_SESSION['current_company_id']) || $_SESSION['current_company_id'] != $company['id'])) {
    $_SESSION['current_company_id'] = $company['id'];
    $_SESSION['current_company'] = $company;
}

$companyId = $company['id'] ?? null;
$workspaceSlugValue = $company['username'] ?? ($companyId ? (string) $companyId : null);
$workspaceQuery = $workspaceSlugValue ? '?workspace=' . urlencode((string) $workspaceSlugValue) : '';
$workspaceQueryAmp = $workspaceSlugValue ? '&workspace=' . urlencode((string) $workspaceSlugValue) : '';

// Debug: Log what we're actually using
error_log("SETTINGS - Company: " . ($company['name'] ?? 'none') . " (ID: " . ($company['id'] ?? 'none') . ", Username: " . ($company['username'] ?? 'none') . ")");

// Handle case where company is not found
if (!$companyId) {
    die("No company found. Please contact support.");
}

// Get current user
$user = AuthManager::getCurrentUser();

// Check if user is owner of this workspace
$isOwner = PermissionManager::isOwner((int)$user['id'], (int)$companyId);

// Helper function to generate clean username from company name
function generateCleanUsername($companyName) {
    // Convert to lowercase and replace spaces/special chars with hyphens
    $username = strtolower($companyName);
    $username = preg_replace('/[^a-z0-9\s-]/', '', $username);
    $username = preg_replace('/[\s-]+/', '-', $username);
    $username = trim($username, '-');
    
    // Ensure it's at least 3 characters
    if (strlen($username) < 3) {
        $username = $username . 'ws';
    }
    
    // Limit to 20 characters
    if (strlen($username) > 20) {
        $username = substr($username, 0, 20);
    }
    
    return $username;
}

// Check if current username looks like auto-generated hash (contains random hex-like strings)
if ($company && isset($company['username'])) {
    $currentUsername = $company['username'];
    // If username looks like auto-generated (contains long random strings), suggest a clean one
    if (preg_match('/[a-f0-9]{10,}/i', $currentUsername) || strlen($currentUsername) > 25) {
        $cleanUsername = generateCleanUsername($company['name']);
        // You could optionally update the database here or just show it as default in the form
    }
}

$successMessage = '';
$errors = [];

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'update_workspace') {
        $companyName = trim($_POST['company_name'] ?? '');
        $workspaceUsername = trim($_POST['workspace_username'] ?? '');
        
        // Validation
        if (empty($companyName)) {
            $errors[] = 'Workspace name is required';
        }
        
        if (empty($workspaceUsername)) {
            $errors[] = 'Workspace username is required';
        } elseif (!preg_match('/^[a-zA-Z0-9_-]+$/', $workspaceUsername)) {
            $errors[] = 'Workspace username can only contain letters, numbers, underscores, and hyphens';
        } elseif (strlen($workspaceUsername) < 3) {
            $errors[] = 'Workspace username must be at least 3 characters long';
        } elseif (strlen($workspaceUsername) > 20) {
            $errors[] = 'Workspace username must be less than 20 characters';
        } else {
            // Only check uniqueness if username column exists
            try {
                $stmt = $pdo->prepare("SELECT id FROM companies WHERE username = ? AND id != ?");
                $stmt->execute([$workspaceUsername, $companyId]);
                if ($stmt->fetch()) {
                    $errors[] = 'Workspace username is already taken';
                }
            } catch (Exception $e) {
                // Column doesn't exist yet, skip uniqueness check
            }
        }
        
        if (empty($errors)) {
            try {
                // Check if username column exists, if not add it
                $pdo->exec("ALTER TABLE companies ADD COLUMN username VARCHAR(50) AFTER name");
            } catch (Exception $e) {
                // Column probably already exists
            }
            
            try {
                // Update company with new data (only name and username)
                $stmt = $pdo->prepare("UPDATE companies SET name = ?, username = ?, updated_at = NOW() WHERE id = ?");
                $stmt->execute([$companyName, $workspaceUsername, $companyId]);
                
                // Update session company data
                $_SESSION['current_company']['name'] = $companyName;
                $_SESSION['current_company']['username'] = $workspaceUsername;
                
                // Refresh company data for display
                $company['name'] = $companyName;
                $company['username'] = $workspaceUsername;
                
                $successMessage = 'Workspace settings updated successfully!';
            } catch (Exception $e) {
                $errors[] = 'Failed to update workspace: ' . $e->getMessage();
            }
        }
    } elseif ($action === 'delete_workspace') {
        $confirmName = trim($_POST['confirm_name'] ?? '');
        $confirmUsername = trim($_POST['confirm_username'] ?? '');
        
        if ($confirmName !== $company['name'] || $confirmUsername !== ($company['username'] ?? $company['name'])) {
            $errors[] = 'Workspace name and username do not match. Please confirm correctly.';
        } else {
            try {
                // Start transaction
                $pdo->beginTransaction();
                
                // Capture users tied to this workspace so we can detach them later
                $stmt = $pdo->prepare("SELECT id FROM users WHERE company_id = ?");
                $stmt->execute([$companyId]);
                $affectedUsers = $stmt->fetchAll(PDO::FETCH_COLUMN) ?: [];

                // Build list of tables that reference company_id
                $tablesWithCompanyId = [];
                try {
                    $stmt = $pdo->query("SELECT TABLE_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND COLUMN_NAME = 'company_id'");
                    $tablesWithCompanyId = $stmt ? $stmt->fetchAll(PDO::FETCH_COLUMN) : [];
                } catch (Exception $e) {
                    // Fall back to common tables if INFORMATION_SCHEMA is not accessible
                    $tablesWithCompanyId = ['invoices', 'invoice_items', 'contracts', 'proposals', 'clients', 'templates', 'modules', 'labels', 'subscriptions', 'billing_invoices', 'usage_tracking', 'payment_methods', 'user_company_access'];
                }

                $tablesWithCompanyId = array_unique(array_filter($tablesWithCompanyId, static function ($table) {
                    $table = strtolower($table);
                    return $table !== 'companies' && $table !== 'users';
                }));

                foreach ($tablesWithCompanyId as $table) {
                    try {
                        $stmt = $pdo->prepare("DELETE FROM `{$table}` WHERE company_id = ?");
                        $stmt->execute([$companyId]);
                    } catch (Exception $e) {
                        // Table might not exist or may not have company_id; ignore and continue
                    }
                }

                // Detach users from deleted workspace instead of deleting them
                if (!empty($affectedUsers)) {
                    try {
                        $placeholder = implode(',', array_fill(0, count($affectedUsers), '?'));
                        $stmt = $pdo->prepare("UPDATE users SET company_id = NULL WHERE id IN ($placeholder)");
                        $stmt->execute($affectedUsers);
                    } catch (Exception $e) {
                        // If update fails, continue so deletion can proceed
                    }
                }

                // Delete the company
                $stmt = $pdo->prepare("DELETE FROM companies WHERE id = ?");
                $stmt->execute([$companyId]);

                // Delete workspace folder if it exists
                $workspaceFolder = __DIR__ . '/../storage/companies/' . $companyId;
                deleteWorkspaceDirectory($workspaceFolder);

                $pdo->commit();
                
                // Clear session and redirect to dashboard
                unset($_SESSION['current_company']);
                unset($_SESSION['current_company_id']);
                
                header('Location: /index.php');
                exit;
                
            } catch (Exception $e) {
                $pdo->rollback();
                $errors[] = 'Failed to delete workspace: ' . $e->getMessage();
            }
        }
    }
}

// Get workspace statistics
 $clientCount = DatabaseHelper::safeCount('clients', 'company_id = ?', [$companyId]);
 $templateCount = DatabaseHelper::safeCount('templates', 'company_id = ?', [$companyId]);
 $moduleCount = DatabaseHelper::safeCount('modules', 'company_id = ?', [$companyId]);
 $proposalCount = DatabaseHelper::safeCount('proposals', 'company_id = ?', [$companyId]);
 $contractCount = DatabaseHelper::safeCount('contracts', 'company_id = ?', [$companyId]);
 $invoiceCount = DatabaseHelper::safeCount('invoices', 'company_id = ?', [$companyId]);

 $currentTheme = $_GET['theme'] ?? ($_SESSION['theme_preference'] ?? 'dark');
 $currentTheme = in_array($currentTheme, ['dark', 'light'], true) ? $currentTheme : 'dark';
 $workspaceDisplayName = $company['name'] ?? 'Workspace';

?>
<!DOCTYPE html>
<html lang="en" data-theme="<?= htmlspecialchars($currentTheme) ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Workspace Settings · <?= htmlspecialchars($workspaceDisplayName) ?></title>
    <link rel="icon" type="image/png" href="/assets/favicon.png">
    <link rel="stylesheet" href="/assets/styles/app.build.css">
    <script>
        (function() {
            var saved = localStorage.getItem('theme') || '<?= $currentTheme ?>';
            document.documentElement.setAttribute('data-theme', saved);
        })();
    </script>
</head>
<body class="settings-page">
<?php require_once __DIR__ . '/navigation.php'; renderNavigationNew('Settings', $currentTheme); ?>

<div class="main-container">
    <?php
    require_once __DIR__ . '/components/headers/page-header.php';

    $actions = [
        [
            'label' => 'Members',
            'url' => '/members.php',
            'type' => 'secondary',
            'icon' => 'M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z',
            'append_workspace' => true,
        ],
        [
            'label' => 'Billing',
            'url' => '/billing.php',
            'type' => 'secondary',
            'icon' => 'M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z M15 12a3 3 0 11-6 0 3 3 0 016 0z',
            'append_workspace' => true,
        ],
    ];

    renderPageHeader(
        breadcrumbs: [
            ['label' => 'Home', 'url' => '/index.php', 'icon' => 'M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6'],
        ],
        currentPage: 'Workspace settings',
        actions: $actions,
        workspaceQuery: $workspaceQuery
    );
    ?>

    <div class="dash-container settings-dashboard">
        <div class="dash-section">
            <section class="card" style="margin-bottom: 0;">
                <div class="settings-hero">
                    <div class="settings-hero__top">
                        <div>
                            <p class="section-title">Workspace</p>
                            <h2 class="settings-hero__title"><?= htmlspecialchars($workspaceDisplayName) ?></h2>
                            <p class="settings-hero__subtitle">Manage your workspace profile, URL, and lifecycle settings.</p>
                        </div>
                        <div class="settings-hero__meta">
                            <span class="status-chip">ID <?= (int) $companyId ?></span>
                            <span class="status-chip">Created <?= htmlspecialchars(date('M j, Y', strtotime($company['created_at'] ?? 'now'))) ?></span>
                        </div>
                    </div>
                </div>
            </section>
        </div>

        <div class="dash-section">
            <div class="members-tabs">
                <a class="members-tab is-active" href="/settings.php<?= $workspaceQuery ?>">Workspace settings</a>
                <a class="members-tab" href="/members.php<?= $workspaceQuery ?>">Team</a>
                <a class="members-tab" href="/activity.php<?= $workspaceQuery ?>">Activity log</a>
            </div>
        </div>

        <?php if ($successMessage): ?>
            <div class="dash-section">
                <div class="status-banner success">
                    <span><?= htmlspecialchars($successMessage) ?></span>
                </div>
            </div>
        <?php endif; ?>

        <?php if ($errors): ?>
            <div class="dash-section">
                <div class="status-banner error">
                    <ul class="status-list">
                        <?php foreach ($errors as $error): ?>
                            <li><?= htmlspecialchars($error) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            </div>
        <?php endif; ?>

        <?php if (!$isOwner): ?>
            <div class="dash-section">
                <section class="card" style="padding: 32px; text-align: center;">
                    <p class="section-title">Access</p>
                    <h2 style="font-size: 12px; font-weight: 500; color: var(--text-primary); margin: 4px 0;">Settings not available</h2>
                    <p style="font-size: 11px; color: var(--text-tertiary); margin-top: 4px;">Only the workspace owner can manage these settings.</p>
                    <div style="margin-top: 12px;">
                        <a href="/index.php<?= $workspaceQuery ?>" class="btn-secondary">Back to dashboard</a>
                    </div>
                </section>
            </div>
        <?php else: ?>
            <div class="dash-section">
                <section class="card" style="margin-bottom: 0;">
                    <div style="padding-bottom: 12px; border-bottom: 1px solid var(--border-subtle); margin-bottom: 12px;">
                        <p class="section-title">Snapshot</p>
                        <p class="settings-section-subtitle">Quick totals across your workspace</p>
                    </div>
                    <div class="settings-metrics-grid">
                        <?php
                        $metrics = [
                            ['label' => 'Clients', 'count' => $clientCount, 'hint' => 'Active clients', 'icon' => 'M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z'],
                            ['label' => 'Proposals', 'count' => $proposalCount, 'hint' => 'Generated proposals', 'icon' => 'M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z'],
                            ['label' => 'Contracts', 'count' => $contractCount, 'hint' => 'Signed contracts', 'icon' => 'M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z'],
                            ['label' => 'Invoices', 'count' => $invoiceCount, 'hint' => 'Created invoices', 'icon' => 'M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z'],
                        ];
                        foreach ($metrics as $metric):
                        ?>
                            <div class="settings-metric">
                                <div class="settings-metric__top">
                                    <p class="settings-metric__label"><?= htmlspecialchars($metric['label']) ?></p>
                                    <div class="settings-metric__icon">
                                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="<?= $metric['icon'] ?>"></path>
                                        </svg>
                                    </div>
                                </div>
                                <p class="settings-metric__value"><?= (int) $metric['count'] ?></p>
                                <p class="settings-metric__hint"><?= htmlspecialchars($metric['hint']) ?></p>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </section>
            </div>

            <div class="dash-section">
                <section class="card" style="margin-bottom: 0;">
                    <div style="padding-bottom: 12px; border-bottom: 1px solid var(--border-subtle); margin-bottom: 12px;">
                        <p class="section-title">General settings</p>
                        <p class="settings-section-subtitle">Update the workspace display name and workspace URL</p>
                    </div>

                    <form method="POST" class="settings-form" data-settings-form>
                        <input type="hidden" name="action" value="update_workspace">

                        <div class="form-grid">
                            <div>
                                <label class="form-label" for="company_name">Workspace name</label>
                                <input class="input" type="text" id="company_name" name="company_name" value="<?= htmlspecialchars($company['name'] ?? '') ?>" required>
                                <p class="form-hint">The display name for your workspace.</p>
                            </div>

                            <div>
                                <label class="form-label" for="workspace_username">Workspace username</label>
                                <input class="input" type="text" id="workspace_username" name="workspace_username" value="<?= htmlspecialchars($company['username'] ?? '') ?>" required pattern="[a-zA-Z0-9_-]+" minlength="3" maxlength="20">
                                <p class="form-hint">3-20 characters, using only letters, numbers, underscores, and hyphens.</p>
                                <?php if (isset($company['username']) && (preg_match('/[a-f0-9]{10,}/i', $company['username']) || strlen($company['username']) > 25)):
                                    $cleanUsername = generateCleanUsername($company['name']);
                                ?>
                                    <div class="settings-suggestion">
                                        <p class="form-label" style="margin: 0;">Suggestion</p>
                                        <p class="form-hint" style="margin: 4px 0 0;">Consider using <code><?= htmlspecialchars($cleanUsername) ?></code> for a cleaner URL.</p>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <div class="settings-form-footer">
                            <div class="settings-meta">
                                <p>Workspace ID: <span><?= (int) $companyId ?></span></p>
                                <p>Created: <span><?= htmlspecialchars(date('M j, Y', strtotime($company['created_at'] ?? 'now'))) ?></span></p>
                            </div>
                            <button type="submit" class="btn-primary">
                                <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                                </svg>
                                Save changes
                            </button>
                        </div>
                    </form>
                </section>
            </div>

            <div class="dash-section">
                <section class="card settings-danger-card" style="margin-bottom: 0;">
                    <div style="padding-bottom: 12px; border-bottom: 1px solid rgba(235, 87, 87, 0.15); margin-bottom: 12px;">
                        <p class="section-title">Danger zone</p>
                        <p class="settings-section-subtitle">Deleting a workspace removes all clients, proposals, contracts, invoices, and uploaded files.</p>
                    </div>

                    <form method="POST" onsubmit="return confirm('Are you absolutely sure? This action cannot be undone.')">
                        <input type="hidden" name="action" value="delete_workspace">
                        <div class="settings-danger-grid">
                            <div>
                                <label class="form-label" for="confirm_name">Workspace name</label>
                                <input class="input input-danger" type="text" id="confirm_name" name="confirm_name" placeholder="<?= htmlspecialchars($workspaceDisplayName) ?>" required>
                            </div>
                            <div>
                                <label class="form-label" for="confirm_username">Workspace username</label>
                                <input class="input input-danger" type="text" id="confirm_username" name="confirm_username" placeholder="<?= htmlspecialchars($company['username'] ?? $workspaceDisplayName) ?>" required>
                            </div>
                        </div>
                        <button type="submit" class="btn-danger">
                            <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 15.5c-.77.833.192 2.5 1.732 2.5z"></path>
                            </svg>
                            Delete workspace forever
                        </button>
                    </form>
                </section>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
    var formChanged = false;
    var form = document.querySelector('[data-settings-form]');

    if (form) {
        form.querySelectorAll('input, textarea').forEach(function (input) {
            input.addEventListener('input', function () {
                formChanged = true;
            });
        });

        window.addEventListener('beforeunload', function (e) {
            if (formChanged) {
                e.preventDefault();
                e.returnValue = '';
            }
        });

        form.addEventListener('submit', function () {
            formChanged = false;
        });
    }
});
</script>

</body>
</html>
