#!/usr/bin/env python3
"""
Remove duplicate inline CSS from PHP files that's already in app.css
"""

import os
import re
from pathlib import Path

PUBLIC_DIR = Path("/Users/craiggomes/Documents/Workstation/Projects/Keelance/public")

# Pattern to match inline style blocks with CSS variables and common classes
STYLE_PATTERN = re.compile(
    r'<style>\s*:root\s*{[^}]+}.*?</style>',
    re.DOTALL
)

def clean_inline_styles(file_path):
    """Remove duplicate inline styles from a PHP file"""
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
        
        original_content = content
        
        # Remove inline style blocks that define CSS variables
        # Keep only if they have page-specific styles not in app.css
        if '<style>' in content and ':root {' in content:
            # Check if it's just CSS variables and common classes
            style_match = re.search(r'<style>(.*?)</style>', content, re.DOTALL)
            if style_match:
                style_content = style_match.group(1)
                # If it contains mostly CSS variables and common classes, remove it
                if '--page-bg:' in style_content or '--surface-' in style_content:
                    # Remove the entire style block
                    content = re.sub(
                        r'\t<style>.*?</style>\n',
                        '',
                        content,
                        flags=re.DOTALL
                    )
        
        # Only write if content changed
        if content != original_content:
            with open(file_path, 'w', encoding='utf-8') as f:
                f.write(content)
            return True
        return False
    except Exception as e:
        print(f"  ⚠️  Error processing {file_path.name}: {e}")
        return False

def main():
    """Process all PHP files in public directory"""
    php_files = list(PUBLIC_DIR.glob("*.php"))
    
    print(f"Found {len(php_files)} PHP files to process\n")
    
    updated_count = 0
    for php_file in sorted(php_files):
        if php_file.name in ['navigation.php', 'footer.php']:
            continue  # Skip component files
        
        if clean_inline_styles(php_file):
            print(f"✓ Cleaned: {php_file.name}")
            updated_count += 1
    
    print(f"\n✅ Updated {updated_count} files")
    print("All pages now use compiled CSS from app.build.css")

if __name__ == "__main__":
    main()
